<?php

namespace Clevercherry\MediaManager;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Request;
use Illuminate\Support\Facades\Storage;

class Media extends Model
{

    protected $table = 'media';

    protected $fillable = [
        'category', 'tags', 'type', 'key', 'group', 'ident', 'list', 'path', 'original', 'aspect', 'order', 'filename', 'extension','fields',
    ];

    /**
     * Returns an instance of the media manager
     * @param string $table 
     * @param string $option 
     * @param string|string $title 
     * @param int|null $id 
     * @return type
     */
    public static function manager(string $table, string $title = 'Media Manager', $id = null)
    {
        return view('media-manager::index')->with('table', $table)
                                           ->with('title', $title)
                                           ->with('id', $id);
    }

    /**
     * Returns all media options for a given category
     * @param string $option 
     * @param int|null $id 
     * @return type
     */
    public static function getMediaOptions(string $option, $id = null)
    {
        $return = [];
        $options = config('media.'.$option);

        foreach($options AS $key => $item){

            if(empty($item['include']) && ! in_array($id, $item['exclude'])){
                $return[$key] = $item;
            } elseif(in_array($id, $item['include'])){
                $return[$key] = $item;
            }

        }

        $defaultValues = [
            'ident' => 'undefined',
            'category' => 'Undefined',
            'width' => 0,
            'height' => 0,
            'exclude' => [],
            'include' => [],
            'fill' => '#fff',
            'image' => true,
            'video' => true,
            'fields' => [],
        ];

        foreach($return AS $key => $option) {
            foreach($defaultValues AS $default => $value) {
                if(! isset($option[$default])) {
                    $return[$key][$default] = $value;
                }
            }
        }

        return $return;
    }

    /**
     * Returns a media config option within a given category
     * @param string $option 
     * @param string $category 
     * @return type
     */
    public static function getMediaCategory(string $option, string $category)
    {
        $options = config('media.'.$option);

        $return = $options[$category];

        $defaultValues = [
            'ident' => 'undefined',
            'category' => 'Undefined',
            'width' => 0,
            'height' => 0,
            'exclude' => [],
            'include' => [],
            'fill' => '#fff',
            'image' => true,
            'video' => true,
            'fields' => [],
        ];

        foreach($defaultValues AS $key => $value) {
            if(! isset($return[$key])) {
                $return[$key] = $value;
            }
        }

        return $return;
    }

    /**
     * Accepts a foreign key and searches the current request object for a group id to associate the group with the foreign key
     * @param int $id 
     * @return type
     */
    public static function couple(int $id)
    {       
        $groups = array_unique(explode(',',Request()->groups));

        if(empty($groups)){
            return true;         
        }

        try{
            foreach($groups AS $group){
                Media::where('group', $group)->update(['key' => $id]);
            }
            return true;
        } catch(\Exception $e){
            return false;
        }
    }

    /**
     * Gets the group ID for a given type and ident or creates a new one
     * @param type $type
     * @param type $ident
     * @return type
     */
    public static function getOrCreateGroup($type, $key, $ident)
    {
        if(! empty($key) && $key > 0){
            $media = Media::where([['type', $type],['key', $key],['ident', $ident]])->first();
    
            if(! empty($media->group)){
                return $media->group;
            }
        }

        $latest = Media::orderBy('group', 'desc')->first();

        if(! $latest){
            return 1;
        }

        return $latest->group+1;
    }

    /**
     * Gets the group ID for a given type and ident or returns null of none exists
     * @return type
     */
    public static function getGroupOrNull($type, $key, $ident)
    {
        if(! empty($key) && $key > 0){
            $media = Media::where([['type', $type],['key', $key],['ident', $ident]])->first();
    
            if(! empty($media->group)){
                return $media->group;
            }
        }

        return null;
    }

    /**
     * Gets all media in a given group
     * @param type $variables
     * @param type|null $column
     * @param type|null $value
     * @return type
     */
    public static function getMedia($group = null, $type = null, $key = null, $ident = null)
    {
        if(empty($group) && empty($key)){
            return [];
        }

        if(empty($group)){
            $media = Media::where([['type', $type],['key', $key],['ident', $ident]])->first();
            if(empty($media->group)){
                return [];
            }

            $group = $media->group;
        }

        return Media::where('group', $group)->orderBy('order', 'asc')->get();
    }

    /**
     * Returns the first image in a given group
     * @param type|null $type
     * @param type|null $key
     * @param type|null $ident
     * @return type
     */
    public static function getFirstImageAttribute(string $type, int $key, string $ident, string $attribute)
    {
        $media = Media::where([['type', $type],['key', $key],['ident', $ident]])->first();

        if(! $media) {
            return null;
        }

        $fields = json_decode($media->fields, true);

        foreach($fields AS $key => $field) {
            $media->{$key} = $field;
        }

        return $media->{$attribute};
    }

    public static function getFirstImagePath(string $type, int $key, string $ident)
    {
        $path = self::getFirstImageAttribute($type, $key, $ident, 'path');

        if(empty($path)) {
            return url('images/'.$type.'/'.$ident.'.png');
        }

        return Storage::url($path);
    }

    /**
     * Returns the amount of items in a given group plus one
     * @param int $group
     * @return type
     */
    public static function getNewOrder(int $group)
    {
        $count = Media::where('group', $group)->count();

        return $count+1;
    }

    /**
     * Returns the latest 20 media items by default
     * @param int|null $group
     * @param int|int $limit
     * @return type
     */
    public static function latest($group = null, $type = null, $key = null, $ident = null, $limit = 20)
    {
        if(empty($group)){
            $media = Media::where([['type', $type],['key', $key],['ident', $ident]])->first();

            if(empty($media->group)){
                $group = 0;
            } else{
                $group = $media->group;
            }
        }

        return Media::where('group', '!=', $group)->groupBy('path')->orderBy('created_at', 'desc')->get();
    }

    /**
     * Returns a unique filename within a driectory by checking if the uploaded filename exists and incrementing an appended integer until a unique filename is found
     * @param type $directory
     * @param type $filename
     * @param type $extension
     * @return type
     */
    public static function uniqueName($directory, $filename, $extension)
    {
        $i = 1;
        $unique = $filename;
        $check = true;

        if(is_file(storage_path('app/'.$directory.'/'.$filename.$extension))){
            $check = false;
        }

        while($check == false){
            $unique = $filename.'-'.$i;
            if(! is_file(storage_path('app/'.$directory.'/'.$unique.$extension))){
                $check = true;
            }
            $i = $i+1;
        }

        return $unique;
    }

    /**
     * Counts all images in a given group
     * @param type|null $group
     * @param type|null $type
     * @param type|null $key
     * @param type|null $ident
     * @return type
     */
    public static function countImages($group = null, $type = null, $key = null, $ident = null)
    {
        if(empty($group)){
            $media = Media::where([['type', $type],['key', $key],['ident', $ident]])->first();

            if(empty($media->group)){
                $group = 0;
            } else{
                $group = $media->group;
            }
        }

        return Media::where('group', $group)->whereNotIn('extension', Media::videoExtensions())->count();
    }

    /**
     * Counts all videos in a given group
     * @param type|null $group
     * @param type|null $type
     * @param type|null $key
     * @param type|null $ident
     * @return type
     */
    public static function countVideos($group = null, $type = null, $key = null, $ident = null)
    {
        if(empty($group)){
            $media = Media::where([['type', $type],['key', $key],['ident', $ident]])->first();

            if(empty($media->group)){
                $group = 0;
            } else{
                $group = $media->group;
            }
        }

        return Media::where('group', $group)->whereIn('extension', Media::videoExtensions())->count();
    }

    /**
     * Counts all media in a given group
     * @param type|null $group
     * @param type|null $type
     * @param type|null $key
     * @param type|null $ident
     * @return type
     */
    public static function countAll($group = null, $type = null, $key = null, $ident = null)
    {
        if(empty($group)){
            $media = Media::where([['type', $type],['key', $key],['ident', $ident]])->first();

            if(empty($media->group)){
                $group = 0;
            } else{
                $group = $media->group;
            }
        }

        return Media::where('group', $group)->count();
    }

    /**
     * Checks if a given string has a video extension
     * @param type $item
     * @return type
     */
    public static function isVideo($item)
    {
        $segments = explode('.', $item);
        $ext = end($segments);

        if(in_array($ext, Media::videoExtensions())){
            return true;
        }

        return false;
    }

    /**
     * Returns an array of video file extensions
     * @return type
     */
    public static function videoExtensions()
    {
        return [
            'mp4', 'mov',
        ];
    }
}
