<?php

namespace Clevercherry\MediaManager\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\Facades\Image;
use Illuminate\Support\Str;
use Clevercherry\MediaManager\Media;

class MediaController extends Controller
{
    /**
     * Handles the upload or update of an image
     * @param Request $request 
     * @return type
     */
    public function image(Request $request)
    {
        try{
            $request->filename = Str::slug($request->filename);
            if(! empty($request->croppedImage)){
                $path = $this->upload('public/images', $request->croppedImage, $request->filename);
                $original = $this->upload('public/images/original', $request->originalImage, $request->filename);
                $originalPath = $original->path;
            } else{
                $path = $this->upload('public/original', $request->originalImage, $request->filename);
                $originalPath = null;
            }

            $store = [];
            $store['category'] = $request->category;
            $store['tags'] = $request->tags;
            $store['path'] = $path->path;
            $store['original'] = $originalPath;
            $store['aspect'] = $path->aspect;
            $store['type'] = $request->type;
            $store['key'] = $request->key;
            $store['ident'] = $request->ident;
            $store['alt'] = $request->alt;
            $store['caption'] = $request->caption;
            $store['header'] = $request->header;
            $store['sub_header'] = $request->sub_header;
            $store['header_link'] = $request->header_link;
            $store['header_link_text'] = $request->header_link_text;
            $store['filename'] = $path->filename;
            $store['extension'] = str_replace('.','',$path->extension);

            (! empty($request->group) && $request->group!=='undefined') ? $store['group'] = $request->group : $store['group'] = Media::getOrCreateGroup($store['type'], $store['key'], $store['ident']);

            $store['order'] = Media::getNewOrder($store['group']);
            (! empty($request->list)) ? $store['list'] = 1 : $store['list'] = 0;


            $media = Media::create($store);
            $media = Media::getMedia($store['group']);
            $value = Media::getMediaCategory($store['category'], $store['ident']);

            return $store['group'];

        } catch(\Exception $e){
            return false;
        }
    }

    /**
     * Handles the upload of a video
     * @param Request $request 
     * @return type
     */
    public function video(Request $request)
    {
        try{
            $request->video_name = Str::slug($request->video_name);
            $extension = $request->file('video')->getClientOriginalExtension();
            $filename = Media::uniqueName(storage_path('app/public/videos'), $request->video_name, '.'.$extension);
            $path = Storage::putFileAs('public/videos', $request->file('video'), $filename.'.'.$extension);

            $store = [];
            $store['category'] = $request->category;
            $store['tags'] = $request->tags;
            $store['path'] = $path;
            $store['type'] = $request->type;
            $store['key'] = $request->key;
            $store['ident'] = $request->ident;
            $store['alt'] = $request->alt;
            $store['caption'] = $request->caption;
            $store['header'] = $request->header;
            $store['sub_header'] = $request->sub_header;
            $store['header_link'] = $request->header_link;
            $store['header_link_text'] = $request->header_link_text;
            $store['filename'] = $filename.'.'.$extension;
            $store['extension'] = $extension;

            (! empty($request->group)) ? $store['group'] = $request->group : $store['group'] = Media::getOrCreateGroup($store['type'], $store['key'], $store['ident']);
            $store['order'] = Media::getNewOrder($store['group']);
            (! empty($request->list)) ? $store['list'] = 1 : $store['list'] = 0;

            $media = Media::create($store);
            $media = Media::getMedia($store['group']);
            $value = Media::getMediaCategory($store['category'], $store['ident']);

            return $store['group'];

        } catch(\Exception $e){
            return false;
        }

    }

    /**
     * Process the image upload from a blob
     * @param type $path 
     * @param type $blob 
     * @param type $name 
     * @return type
     */
    public function upload($path, $blob, $name)
    {
        if(! is_dir(storage_path('app/'.$path))){
            mkdir(storage_path('app/'.$path));
        }

        $store = [];
        $pos  = strpos($blob, ';');
        $type = explode(':', substr($blob, 0, $pos))[1];
        $extension = str_replace('image/', '.', $type);
        $filename = Media::uniqueName($path, $name, $extension);
        $filename = $filename.$extension;

        $image = Image::make($blob);
        $height = $image->height();
        $width = $image->width();
        $image->save(storage_path('app/'.$path.'/'.$filename));

        $return = new \stdClass();
        $return->path = $path.'/'.$filename;
        $return->aspect = number_format($height/$width,2);
        $return->filename = $filename;
        $return->extension = $extension;

        return $return;
    }

    /**
     * Assigns media to new types, keys, idents and groups and reorders the media set by the user
     * @param Request $request 
     * @return type
     */
    public function reorder(Request $request)
    {

        foreach($request->data AS $data){
            $type = $data['type'];
            $key = $data['key'];
            $ident = $data['ident'];
            $category = $data['category'];

            if(! isset($data['group'])){
                return 'No Images or Videos';
            }

            $group = $data['group'];

            if(empty($data['group']) || $data['group']=='undefined'){
                $group = Media::getOrCreateGroup($type, $key, $ident);
            }

            if(! isset($data['media'])){
                $media = Media::getMedia($group);
                foreach($media AS $value){
                    $data['media'][] = $value->id;
                }
            }

            $ids = [];

            if(isset($data['media'])){            
                foreach($data['media'] AS $item){

                    $media = Media::where('id', $item)->first();

                    if($media->type==$type && $media->key==$key && $media->ident==$ident){
                        $ids[] = $media->id;
                    } else{
                        $store = [];
                        $store['category'] = $media->category;
                        $store['tags'] = $media->tags;
                        $store['type'] = $type;
                        $store['key'] = $key;
                        $store['group'] = $group;
                        $store['ident'] = $ident;
                        $store['alt'] = $media->alt;
                        $store['caption'] = $media->caption;
                        $store['header'] = $request->header;
                        $store['sub_header'] = $request->sub_header;
                        $store['header_link'] = $request->header_link;
                        $store['header_link_text'] = $request->header_link_text;
                        $store['path'] = $media->path;
                        $store['original'] = $media->original;
                        $store['aspect'] = $media->aspect;
                        $store['filename'] = $media->filename;
                        $store['extension'] = $media->extension;

                        $media = Media::create($store);
                        $ids[] = $media->id;
                    }
                }
            }

            $entries = Media::where([['type', $type],['key', $key],['ident', $ident]])->get();

            foreach($entries AS $entry){
                if(! in_array($entry->id, $ids)){
                    Media::where('id', $entry->id)->update(['type' => null, 'key' => null, 'group' => null, 'ident' => null]);
                }
            }

            foreach($ids AS $order => $id){
                Media::where('id', $id)->update(['order' => $order]);
            }

        }

        $media = Media::getMedia($group);
        $value = Media::getMediaCategory($category, $ident);

        return view('media-manager::partials.media-refresh-gallery')->with('media', $media)
                                                                    ->with('value', $value)
                                                                    ->with('group', $group);
    }

    /**
     * Refrehsed the gallery in a given group
     * @param int $group 
     * @return type
     */
    public function refreshGroup(int $group, $category, $ident)
    {
        $media = Media::getMedia($group);
        $value = Media::getMediaCategory($category, $ident);
        return view('media-manager::partials.media-refresh-gallery')->with('media', $media)
                                                                    ->with('value', $value)
                                                                    ->with('group', $group);
    }

    /**
     * Updates the information stored about a media item
     * @param int $id 
     * @param Request $request 
     * @return type
     */
    public function update(int $id, Request $request)
    {
        try{
            $store = [];
            $store['category'] = $request->category;
            $store['tags'] = $request->tags;
            $store['alt'] = $request->alt;
            $store['caption'] = $request->caption;
            $store['header'] = $request->header;
            $store['sub_header'] = $request->sub_header;
            $store['header_link'] = $request->header_link;
            $store['header_link_text'] = $request->header_link_text;

            Media::where('id', $id)->update($store);

            return 'success';
        } catch(\Exception $e){
            abort(404);
        }
    }
    
}
