<?php

namespace Clevercherry\MediaManager;

use Illuminate\Database\Eloquent\Model;

class Media extends Model
{

    protected $table = 'media';

    protected $fillable = [
		'category', 'tags', 'type', 'key', 'group', 'ident', 'alt', 'caption', 'list', 'path', 'original', 'aspect', 'order', 'filename', 'extension', 'header', 'sub_header', 'header_link', 'header_link_text',
    ];

    /**
     * Returns an instance of the media manager
     * @param string $table 
     * @param string $option 
     * @param string|string $title 
     * @param int|null $id 
     * @return type
     */
    public static function manager(string $table, string $option, string $title = 'Media Manager', int $id = null)
    {
    	return view('media::index')->with('table', $table)
    							   ->with('option', $option)
    							   ->with('title', $title)
    							   ->with('id', $id);
    }

    /**
     * Returns all media options for a given category
     * @param string $option 
     * @param int|null $id 
     * @return type
     */
    public static function getMediaOptions(string $option, int $id = null)
    {
        $return = [];
    	$options = config('media.'.$option);

        foreach($options AS $key => $item){

            if(empty($item['include']) && ! in_array($id, $item['exclude'])){
                $return[$key] = $item;
            } elseif(in_array($id, $item['include'])){
                $return[$key] = $item;
            }

        }

    	return $return;
    }

    /**
     * Returns a media config option within a given category
     * @param string $option 
     * @param string $category 
     * @return type
     */
    public static function getMediaCategory(string $option, string $category)
    {
        $options = config('media.'.$option);

        return $options[$category];
    }

    /**
     * Gets the group ID for a given type and ident or creates a new one
     * @param type $type
     * @param type $ident
     * @return type
     */
    public static function getOrCreateGroup($type, $key, $ident)
    {
        $media = Media::where([['type', $type],['key', $key],['ident', $ident]])->first();

        if(! empty($media->group)){
            return $media->group;
        }

        $latest = Media::orderBy('group', 'desc')->first();

        if(! $latest){
            return 1;
        }

        return $latest->group+1;
    }

    /**
     * Gets all media in a given group
     * @param type $variables
     * @param type|null $column
     * @param type|null $value
     * @return type
     */
    public static function getMedia($group = null, $type = null, $key = null, $ident = null)
    {
        if(empty($group)){
            $media = Media::where([['type', $type],['key', $key],['ident', $ident]])->first();
            if(empty($media->group)){
                return [];
            }

            $group = $media->group;
        }

        return Media::where('group', $group)->orderBy('order', 'asc')->get();
    }

    /**
     * Returns the first image in a given group
     * @param type|null $group
     * @param type|null $type
     * @param type|null $key
     * @param type|null $ident
     * @return type
     */
    public static function getFirstImage($group = null, $type = null, $key = null, $ident = null)
    {
        try{
            if(empty($group)){
                $media = Media::where([['type', $type],['key', $key],['ident', $ident]])->first();
                if(empty($media->group)){
                    return url('media/development-placeholder.png');
                }

                $group = $media->group;
            }

            return  addcslashes(url(Media::where('group', $group)->whereNotIn('extension', Media::videoExtensions())->orderBy('order', 'asc')->first()->path), "'");
        } catch(\Exception $e){
            return url('media/development-placeholder.png');
        }
    }

    /**
     * Returns the amount of items in a given group plus one
     * @param int $group
     * @return type
     */
    public static function getNewOrder(int $group)
    {
        $count = Media::where('group', $group)->count();

        return $count+1;
    }

    /**
     * Returns the latest 20 media items by default
     * @param int|null $group
     * @param int|int $limit
     * @return type
     */
    public static function latest($group = null, $type = null, $key = null, $ident = null, $limit = 20)
    {
        if(empty($group)){
            $media = Media::where([['type', $type],['key', $key],['ident', $ident]])->first();

            if(empty($media->group)){
                $group = 0;
            } else{
                $group = $media->group;
            }
        }

        return Media::where('group', '!=', $group)->groupBy('path')->orderBy('created_at', 'desc')->get();
    }

    /**
     * Returns the latest 20 media items by default
     * @param int $height
     * @param int $width
     * @return type
     */
    public static function relevant(int $height, int $width)
    {
        try{

            $aspect = $height/$width;

            $min = $aspect*0.9;
            $max = $aspect*1.1;

            return Media::where([['aspect', '>=', $min],['aspect', '<=', $max]])->groupBy('path')->inRandomOrder()->limit(20)->get();
        } catch(\Exception $e){
            return [];
        }
    }

    /**
     * Returns a unique filename within a driectory by checking if the uploaded filename exists and incrementing an appended integer until a unique filename is found
     * @param type $directory
     * @param type $filename
     * @param type $extension
     * @return type
     */
    public static function uniqueName($directory, $filename, $extension)
    {
        $i = 1;
        $unique = $filename;
        $check = true;

        if(is_file(storage_path('app/'.$directory.'/'.$filename.$extension))){
            $check = false;
        }

        while($check == false){
            $unique = $filename.'-'.$i;
            if(! is_file(storage_path('app/'.$directory.'/'.$unique.$extension))){
                $check = true;
            }
            $i = $i+1;
        }

        return $unique;
    }

    /**
     * Counts all images in a given group
     * @param type|null $group
     * @param type|null $type
     * @param type|null $key
     * @param type|null $ident
     * @return type
     */
    public static function countImages($group = null, $type = null, $key = null, $ident = null)
    {
        if(empty($group)){
            $media = Media::where([['type', $type],['key', $key],['ident', $ident]])->first();

            if(empty($media->group)){
                $group = 0;
            } else{
                $group = $media->group;
            }
        }

        return Media::where('group', $group)->whereNotIn('extension', Media::videoExtensions())->count();
    }

    /**
     * Counts all videos in a given group
     * @param type|null $group
     * @param type|null $type
     * @param type|null $key
     * @param type|null $ident
     * @return type
     */
    public static function countVideos($group = null, $type = null, $key = null, $ident = null)
    {
        if(empty($group)){
            $media = Media::where([['type', $type],['key', $key],['ident', $ident]])->first();

            if(empty($media->group)){
                $group = 0;
            } else{
                $group = $media->group;
            }
        }

        return Media::where('group', $group)->whereIn('extension', Media::videoExtensions())->count();
    }

    /**
     * Counts all media in a given group
     * @param type|null $group
     * @param type|null $type
     * @param type|null $key
     * @param type|null $ident
     * @return type
     */
    public static function countAll($group = null, $type = null, $key = null, $ident = null)
    {
        if(empty($group)){
            $media = Media::where([['type', $type],['key', $key],['ident', $ident]])->first();

            if(empty($media->group)){
                $group = 0;
            } else{
                $group = $media->group;
            }
        }

        return Media::where('group', $group)->count();
    }

    /**
     * Checks if a given string has a video extension
     * @param type $item
     * @return type
     */
    public static function isVideo($item)
    {
        $segments = explode('.', $item);
        $ext = end($segments);

        if(in_array($ext, Media::videoExtensions())){
            return true;
        }

        return false;
    }

    /**
     * Returns an array of video file extensions
     * @return type
     */
    public static function videoExtensions()
    {
        return [
            'mp4', 'mov',
        ];
    }
}