<?php

namespace App\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Mail;

use App\Http\Controllers\Controller;

use App\Models\Account;
use App\Models\Product;
use App\Models\ProductCategory;
use App\MediaModel as MediaCustom;

use Media;

class ProductsController extends Controller
{
    public $view = 'admin.crud.products.';
    public $url = '/admin/products';


    /**
     * Display a listing of the resource.
     *
     * @return view
     */
    public function index(Request $request)
    {
        return view($this->view . 'index');
    }

    /**
     * Show the products belonging to a category
     *
     * @return view
     */
    public function view(int $id)
    {
        if($id != 0) {
            $prodCat = ProductCategory::where('id', $id)->first();
        } else {
            $prodCat = 0;
        }
        return view($this->view . 'view')
            ->with('category', $prodCat);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return view
     */
    public function create()
    {
        return view($this->view . 'create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            $item = $request->except('_token');

            if (!isset($item['content'])) {
                return redirect()->back()
                    ->withInput($request->input())
                    ->with('danger', 'Please specifiy content for the page');
            }

            $uniqueTitle = Product::where('name', $item['name'])->first();
            if ($uniqueTitle) {
                return redirect()->back()
                    ->withInput($request->input())
                    ->with('danger', 'A product with this name already exists!');
            }

            $item['slug'] = Str::slug($item['name']);

            // Seo
            if (!isset($item['seo_title'])) {
                $item['seo_title'] = $item['name'];
            }

            $item = Product::create($item);

            Media::couple($item->id);

            return redirect('admin/products/prod-cat/'. $item->product_category)->with('success', 'Record successfully created');
        } catch (\Exception $e) {
            error_log(json_encode($e));
            return redirect()->back()->withInput($request->input())->with('danger', 'An unknown error has occured, please try again and contact your account manager if this error persists');
        }
    }

    /**
     * Loads the update form for an existing record
     * @param type $id
     * @return type
     */
    public function show(int $id)
    {
        $item = Product::where('id', $id)->first();
        if (!$item) {
            return redirect($this->url)->with('danger', 'Sorry, that record could not be found');
        }

        return view($this->view.'update')
            ->with('item', $item);
    }

    /**
     * Stores or updates a product
     * @param int $id - product id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, int $id)
    {
        try {
            $prd = Product::where('id', $id)->first();
            if (!$prd) {
                return redirect()->back()->with('danger', 'Sorry, that record could not be found');
            }

            $store = $request->except('_token');
            // Content
            if (!isset($store['content'])) {
                return redirect()->back()->with('danger', 'Please specifiy content for the page');
            }

            $uniqueTitle = Product::where('name', $store['name'])
                ->where('id', '!=', $id)
                ->first();
            if ($uniqueTitle) {
                return redirect()->back()
                    ->withInput($request->input())
                    ->with('danger', 'A product with this name already exists!');
            }

            // Seo
            if (!isset($store['seo_title'])) {
                $store['seo_title'] = $store['name'];
            }

            $store['slug'] = Str::slug($store['name']);
            $prd->update($store);

            return redirect('admin/products/prod-cat/' . $prd->product_category)->with('success', 'Record successfully updated');
        } catch (\Exception $e) {
            error_log(json_encode($e));
            return redirect('admin/products/prod-cat/' . $prd->product_category)->with('danger', 'An unknown error has occured, please try again and contact your account manager if this error persists');
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     * @return \Illuminate\Http\Response
     */
    public function delete(int $id)
    {
        try {
            $prd = Product::where('id', $id)->first();
            if (!$prd) {
                return redirect()->back()->with('danger', 'Sorry, that record could not be found');
            }

            $prd->delete($prd);

            return ['success' => 'Record successfully removed'];
        } catch (\Exception $e) {
            error_log(json_encode($e));
            return ['error', 'An unknown error has occured, please try again and contact your account manager if this error persists'];
        }
    }


    /**
     * Reorder products
     * @param  Request
     * @return void
     */
    public function reorder(Request $request)
    {
        if (!empty($request->ids)) {
            foreach ($request->ids as $key => $value) {
                $prd = Product::where('id', $value)->first();
                $prd->update(['order' => $key]);
            }
        }
    }
}
