<?php

namespace App\Http\Controllers\Admin;

use App\News as Model;

use Illuminate\Http\Request;
use Illuminate\Support\Str;

use App\Http\Controllers\Controller;

use Media;

class NewsController extends Controller
{
    /**
     * @var directory path for the crud views
     */
    public $view = 'admin.crud.news.';
  
    /**
     * @var admin paths and URLs
     */
    public $adminUrl = '/admin/news';
    public $adminPath = 'admin.view';
   
    /**
     * Returns the index page view
     * @return type
     */
    public function index()
    {
        return view($this->view.'index');
    }
   
    /**
     * Returns the form to create a new record
     * @return type
     */
    public function create()
    {
        return view($this->view.'create');
    }
   
    /**
     * Stores a new page
     * @param Request $request
     * @return type
     */
    public function store(Request $request)
    {
        try {
            $store = $request->except('_token');

            // Content
            if (!isset($store['intro_content'])) {
                return redirect()->back()
                    ->withInput($request->input())
                    ->with('danger', 'Please specifiy content for the article');
            }
            
            $title = Model::where('title', $store['title'])->first();
            if($title) {
                return redirect()->back()
                    ->withInput($request->input())
                    ->with('danger', 'A News Article with this title already exists.');
            }

            // Slug and Parent
            $store['slug'] = Str::slug($store['title']);
            if (!empty($store['parent'])) {
                $parent = Model::where('id', $store['parent'])->first();
                $store['slug'] = $parent->slug.'/'.$store['slug'];
            }
   
            // Seo
            if (! isset($store['seo_title'])) {
                $store['seo_title'] = $store['title'];
            }
   
            $item = Model::create($store);
            // add the media 
            Media::couple($item->id);
            return redirect($this->adminUrl)->with('success', 'Record successfully updated');
        } catch (\Exception $e) {
            return redirect()->back()->with('danger', 'An unknown error has occured, please try again and contact your account manager if this error persists');
        }
    }
  
    /**
     * Loads the update form for an existing record
     * @param type $id
     * @return type
     */
    public function show(int $id)
    {
        $item = Model::where('id', $id)->first();
        if (!$item) {
            return redirect($this->adminUrl)->with('danger', 'Sorry, that record could not be found');
        }
   
        return view($this->view.'update')
            ->with('item', $item);
    }

    /**
     * Stores or updates a news article
     * @param Request $request
     * @return string
     */
    public function update(Request $request, int $id)
    {
        try {
            $page = Model::where('id', $id)->first();
            if (!$page) {
                return redirect()->back()->with('danger', 'Sorry, that record could not be found');
            }

            $store = $request->except('_token');
            
            // Content
            if (!isset($store['intro_content'])) {
                return redirect()->back()->with('danger', 'Please specifiy content for the page');
            }

            // Seo
            if (!isset($store['seo_title'])) {
                $store['seo_title'] = $store['title'];
            }
   
            // Slug and Parent
            $store['slug'] = Str::slug($store['title']);
            if (!empty($store['parent'])) {
                $parent = Model::where('id', $store['parent'])->first();
                $store['slug'] = $parent->slug.'/'.$store['slug'];
            }
   
            $page->update($store);

            return redirect($this->adminUrl)->with('success', 'Record successfully updated');
        } catch (\Exception $e) {
            return redirect($this->adminUrl)->with('danger', 'An unknown error has occured, please try again and contact your account manager if this error persists');
        }
    }

    /**
     * Remove resource
     *
     * @return array response
     */
    public function delete(int $id)
    {
        try {
            $page = Model::where('id', $id)->first();
            if (!$page) {
                return redirect()->back()->with('danger', 'Sorry, that record could not be found');
            }

            // Delete Page
            $page->delete($page);

            return ['success' => 'Record successfully removed'];
        } catch (\Exception $e) {
            return ['error', 'An unknown error has occured, please try again and contact your account manager if this error persists'];
        }
    }
}
