<?php
 
namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Page extends Model
{
    const ACTIVE = 1;
    const INACTIVE = 0;

    const ALLOWS_CHILDREN = 1;
    const NO_CHILDREN = 0;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'pages';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'id', 'title', 'slug', 'seo_title', 'seo_description','heading', 'content', 'secondary_content', 'parent', 'allow_children', 'order', 'active', 'protected'
    ];

    /**
     * Returns an array of all pages statuses
     * @return array
     */
    public static function statuses()
    {
        return array(
            self::ACTIVE => "Active",
            self::INACTIVE => "Inactive"
        );
    }

    /**
     * Returns all protected pages
     * @return array
     */
    public static function protectedPages()
    {
        return Page::whereNull('parent')->orderBy('order', 'ASC')->get();
    }

    /**
     * Returns a protected page by name
     * @param string $name
     * @return row
     */
    public static function getProtectedByName($name)
    {
        return Page::whereNull('parent')
            ->where('protected', $name)
            ->orderBy('created_at', 'ASC')
            ->first();
    }

    /**
     * Returns all child pages of a given page id
     * @param int $id
     * @return array
     */
    public static function children(int $id)
    {
        return Page::where('parent', $id)->orderBy('order', 'ASC')->get();
    }

    /**
     * Returns all child pages of a given page name
     * @param int $id
     * @return array
     */
    public static function childrenByParentName(string $name)
    {
        $page = Page::where('protected', $name)->orderBy('order', 'ASC')->first();
        $children = self::children($page->id);
        return $children ?: array();
    }

    /**
     * Get a list of possible parents, excludes the current pages ID where applicable
     * @param int|null $id
     * @return row
     */
    public static function parentList()
    {
        $array = Page::where('allow_children', '=', self::ALLOWS_CHILDREN)
            ->whereNull('parent')
            ->orderBy('title', 'asc')
            ->get();
        return $array;
    }

    /**
     * Returns a page object based on id
     * @param int $id
     * @return row|empty
     */
    public static function id(int $id)
    {
        return Page::where('id', $id)->first();
    }

    /**
     * Returns a page object based on slug
     * @param string $slug
     * @return row|empty
     */
    public static function slug(string $slug)
    {
        return Page::where('slug', $slug)->first();
    }
}
