<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Product extends Model
{
    // Statuses
    const PENDING = 0;
    const ACTIVE = 1;
    const SOLD = 2;
    const INACTIVE = 3;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'products';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'product_category', 'name', 'code', 'content', 'seo_title', 'seo_description', 'status', 'slug', 'order', 'price'
    ];

    /**
     * Returns the product statuses
     * 
     * @return array
     */
    public static function statuses()
    {
        return array(
            self::PENDING => "Pending",
            self::ACTIVE => "Active",
            self::SOLD => "Sold",
            self::INACTIVE => "Inactive"
        );
    }
    
    /**
     * Returns a of all products
     * @return array
     */
    public static function getAll()
    {
        return self::where('status', self::ACTIVE)
            ->orderBy('created_at', 'desc')->get();
    }


    /**
     * Returns a product slug
     * @param int $id
     * @return type
     */
    public static function slug(string $slug)
    {
        return self::where('slug', $slug)->first();
    }

    /**
     * Returns a list of all products inside the given category
     * @param int $prodCatId - category id
     * @return array
     */
    public static function getAllForCategory($prodCatId)
    {
        $prod = self::where('product_category', $prodCatId)
            ->orderBy('order', 'asc')->get();
        return $prod;
    }

    /**
     * Returns a list of all products inside the given category
     * @param int $prodCatId - category id
     * @return array
     */
    public static function getAllActiveForCategory($prodCatId, $except = 0)
    {
        $prod = self::where('product_category', $prodCatId)
            ->where('status', self::ACTIVE)
            ->where('id', '!=', $except)
            ->orderBy('order', 'asc')->get();
        return $prod;
    }

    /**
     * Returns the count of products inside the given category
     * @param int $prodCatId - category id
     * @return array
     */
    public static function countActiveProductsForCategory($prodCatId)
    {
        $prod = self::where('product_category', $prodCatId)
            ->where(function($q) {
              $q->where('status', self::ACTIVE)
                ->orWhere('status', self::UNDER_OFFER);
            })
            ->get();
        return count($prod);
    }
}
