<?php

namespace App\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Illuminate\Support\Str;

use App\Http\Controllers\Controller;

use App\Models\ProductCategory;
use App\Models\Product;

use Media;

class ProductCategoriesController extends Controller
{
    public $view = 'admin.crud.product-categories.';
    public $url = '/admin/product-categories';

    /**
     * Returns a view with all the product categories
     *
     * @return view
     */
    public function index()
    {
        return view($this->view . 'index');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return view
     */
    public function create()
    {
        return view($this->view . 'create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            $item = $request->except('_token');

            if (!isset($item['content'])) {
                return redirect()->back()
                    ->withInput($request->input())
                    ->with('danger', 'Please specifiy content for the page');
            }

            $uniqueTitle = ProductCategory::where('title', $item['title'])->first();
            if ($uniqueTitle) {
                return redirect()->back()
                    ->withInput($request->input())
                    ->with('danger', 'A Product Category with this name already exists.');
            }

            $item['slug'] = Str::slug($item['title']);

            // Seo
            if (!isset($item['seo_title'])) {
                $item['seo_title'] = $item['title'];
            }

            $item = ProductCategory::create($item);
            Media::couple($item->id);
            return redirect($this->url)->with('success', 'Record successfully created');
        } catch (\Exception $e) {
            return redirect()->back()->with('danger', 'An unknown error has occured, please try again and contact your account manager if this error persists');
        }
    }

    /**
     * Loads the update form for an existing record
     * @param type $id
     * @return view
     */
    public function show(int $id)
    {
        $item = ProductCategory::where('id', $id)->first();
        if (!$item) {
            return redirect($this->url)->with('danger', 'Sorry, that record could not be found');
        }
 
        return view($this->view.'update')
            ->with('item', $item);
    }

    /**
     * Stores or updates a resource
     * @param Request $request
     * @return string
     */
    public function update(Request $request, int $id)
    {
        try {
            $prd = ProductCategory::where('id', $id)->first();
            if (!$prd) {
                return redirect()->back()->with('danger', 'Sorry, that record could not be found');
            }

            $store = $request->except('_token');
            // Content
            if (!isset($store['content'])) {
                return redirect()->back()->with('danger', 'Please specifiy content for the page');
            }

            $uniqueTitle = ProductCategory::where('title', $store['title'])
                ->where('id', '!=', $id)
                ->first();
            if ($uniqueTitle) {
                return redirect()->back()
                    ->withInput($request->input())
                    ->with('danger', 'A Product Category with this name already exists.');
            }

            // Seo
            if (!isset($store['seo_title'])) {
                $store['seo_title'] = $store['title'];
            }
   
            $store['slug'] = Str::slug($store['title']);
            $prd->update($store);

            return redirect($this->url)->with('success', 'Record successfully updated');
        } catch (\Exception $e) {
            error_log($e);
            return redirect($this->url)->with('danger', 'An unknown error has occured, please try again and contact your account manager if this error persists');
        }
    }
 
    /**
     * Reorders and reassigns product categories after drag and drop in the CMS
     * @param Request $request
     *
     * @return void
     */
    public function reorder(Request $request)
    {
        if (!empty($request->ids)) {
            foreach ($request->ids as $key => $value) {
                $prd = ProductCategory::where('id', $value)->first();
                $prd->update(['order' => $key]);
            }
        }
    }

    /**
     * Remove resource
     *
     * @return array response
     */
    public function delete(int $id)
    {
        try {
            $prd = ProductCategory::where('id', $id)->first();
            if (!$prd) {
                return redirect()->back()->with('danger', 'Sorry, that record could not be found');
            }

            // Move Products In Uncategorised  
            $products = Product::getAllForCategory($id);
            foreach ($products as $key => $value) {
                $product = Product::where('id', $value['id'])->first();
                $product->update(['product_category' => ProductCategory::UNCATEGORISED_ID]);
            }

            // Delete Category
            $prd->delete($prd);

            return ['success' => 'Record successfully removed'];
        } catch (\Exception $e) {
            return ['error', 'An unknown error has occured, please try again and contact your account manager if this error persists'];
        }
    }
}
