<?php

namespace Clevercherry\Backslash;

use Illuminate\Contracts\Config\Repository;
use Illuminate\Contracts\Container\Container;
use Illuminate\Contracts\Events\Dispatcher;
use Illuminate\Contracts\View\Factory;
use Illuminate\Support\ServiceProvider as BaseServiceProvider;
use Clevercherry\Backslash\Console\BackslashInstallCommand;
use Clevercherry\Backslash\Console\BackslashPluginCommand;
use Clevercherry\Backslash\Console\BackslashStatusCommand;
use Clevercherry\Backslash\Console\BackslashUpdateCommand;
use Clevercherry\Backslash\Events\BuildingMenu;
use Clevercherry\Backslash\Http\ViewComposers\BackslashComposer;

class BackslashServiceProvider extends BaseServiceProvider
{
    public function register()
    {
        $this->app->singleton(Backslash::class, function (Container $app) {
            return new Backslash(
                $app['config']['backslash.filters'],
                $app['events'],
                $app
            );
        });
    }

    public function boot(
        Factory $view,
        Dispatcher $events,
        Repository $config
    ) {
        $this->loadViews();

        $this->loadTranslations();

        $this->loadConfig();

        $this->registerCommands();

        $this->registerViewComposers($view);

        static::registerMenu($events, $config);
    }

    private function loadViews()
    {
        $resourceViewsPath = $this->packagePath('resources/views');

        if (file_exists($resourceViewsPath)) {
            $viewsPath = $resourceViewsPath;
        } else {
            $viewsPath = $this->packagePath('resources/views');
        }

        $this->loadViewsFrom($viewsPath, 'backslash');
    }

    private function loadTranslations()
    {
        $translationsPath = $this->packagePath('resources/lang');

        $this->loadTranslationsFrom($translationsPath, 'backslash');
    }

    private function loadConfig()
    {
        $configPath = $this->packagePath('config/backslash.php');

        $this->mergeConfigFrom($configPath, 'backslash');
    }

    private function packagePath($path)
    {
        return __DIR__."/../$path";
    }

    private function registerCommands()
    {
        $this->commands(BackslashInstallCommand::class);
        $this->commands(BackslashStatusCommand::class);
        $this->commands(BackslashUpdateCommand::class);
        $this->commands(BackslashPluginCommand::class);
    }

    private function registerViewComposers(Factory $view)
    {
        $view->composer('backslash::page', BackslashComposer::class);
    }

    public static function registerMenu(Dispatcher $events, Repository $config)
    {
        $events->listen(BuildingMenu::class, function (BuildingMenu $event) use ($config) {
            $menu = $config->get('backslash.menu');
            call_user_func_array([$event->menu, 'add'], $menu);
        });
    }
}
